package com.luca.pilot.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import com.luca.pilot.bizEnum.CrfQuestionTypeEnum;
import com.luca.pilot.bo.*;

import java.util.*;
import java.util.stream.Collectors;

public class CrfUtil {

    public static String parseCrfProblemAndAnswer(String crfProblemJson, String crfAswerJson) {
        if (Objects.isNull(crfProblemJson) || "".equals(crfProblemJson)
                || Objects.isNull(crfAswerJson) || "".equals(crfAswerJson)) {
            return "";
        }

        List<CrfProblemBo> crfProblemBos = crfProblemJsonAnalysisNew(crfProblemJson);

        List<String> dataIds = crfProblemBos.stream().map(CrfProblemBo::getDataBo).filter(Objects::nonNull).collect(Collectors.toList())
                .stream().map(BindDataBo::getDataId).collect(Collectors.toList());

        List<CrfJsonValueBo> crfJsonValueBos = crfJsonAnalysis(crfAswerJson);

        List<String> crfAswers = new ArrayList<>();
        crfJsonValueBos.stream().filter(x -> dataIds.contains(x.getKey())).forEach(item -> {
            if (Objects.nonNull(item.getName()) && Objects.nonNull(item.getText())) {
                crfAswers.add(item.getName() + ":" + item.getText());
            }
        });
        if (CollectionUtil.isEmpty(crfAswers)) {
            return null;
        }
        return JSONUtil.toJsonStr(crfAswers);
    }

    /**
     * crf 问题json 解析
     */
    public static List<CrfProblemBo> crfProblemJsonAnalysisNew(String crfProblemJson) {
        if (Objects.isNull(crfProblemJson) || "".equals(crfProblemJson)) {
            return Collections.emptyList();
        }
        var jsonObject = JSONObject.parseObject(crfProblemJson, Feature.OrderedField);
        if (jsonObject == null) {
            return Collections.emptyList();
        }
        List<JSONObject> bos = new ArrayList<>();
        problemJsonNew(bos, jsonObject, null);

        if (CollectionUtil.isEmpty(bos)) {
            return Collections.emptyList();
        }
        return bos.stream().filter(item -> Objects.nonNull(item.get("questionType"))).map(item -> {
            var crfProblemBo = new CrfProblemBo()
                    .setTitle(String.valueOf(item.get("title")))
                    .setProperties(JSONObject.toJSONString(item.get("properties")))
                    .setQuestionType(CrfQuestionTypeEnum.getDeviceEnum(JSONObject.toJSONString(item.get("questionType")).replace("\"", "")));
            var bindData = JSONObject.toJSONString(item.get("BindData"));
            if (!"".equals(bindData)) {
                crfProblemBo.setDataBo(
                        new BindDataBo()
                                .setDataId(JSONObject.parseObject(bindData).getString("dataId"))
                                .setDataType(JSONObject.parseObject(bindData).getString("dataType"))
                                .setChoiceField(JSONObject.parseObject(bindData).getString("choiceField"))
                );
            }
            return crfProblemBo;
        }).collect(Collectors.toList());
    }

    private static void problemJsonNew(List<JSONObject> array, JSONObject data, String prefix) {
        if (Objects.isNull(data)) {
            return;
        }

        var title = data.get("title");
        if (Objects.nonNull(prefix)) {
            title = prefix + "," + title;
            data.put("title", title);

        }
        var properties = data.get("properties");
        if (Objects.isNull(properties)) {
            var columns = data.get("columns");
            var defaultValue = data.get("defaultValue");
            if (Objects.isNull(defaultValue)) {
                array.add(data);
                return;
            }
            try {
                var head = JSONObject.parseObject(data.get("BindData").toString(), Feature.OrderedField).get("dataId").toString();
                var columnsBos = JSONObject.parseArray(JSONObject.toJSONString(columns), CrfTableJsonTopicBo.class).stream()
                        .filter(bo -> !"title".equals(bo.getDataIndex())) // 去掉最左一列 dataIndex 为 title 的列
                        .collect(Collectors.toList());
                var defaultValueBos = JSONObject.parseArray(JSONObject.toJSONString(defaultValue), CrfTableJsonTopicBo.class);
                for (int i = 0; i < columnsBos.size(); i++) {
                    for (int j = 0; j < defaultValueBos.size(); j++) {
                        JSONObject jsonObject = new JSONObject();
                        jsonObject.put("title", title + "," + defaultValueBos.get(j).getTitle() + "-" + columnsBos.get(i).getTitle());
                        jsonObject.put("questionType", CrfQuestionTypeEnum.CRFTABLE.getCode());
                        jsonObject.put("BindData",
                                new BindDataBo()
                                        .setChoiceField("MH")
                                        .setDataType("CUSTOMIZE")
                                        .setDataId(head + "-" + columnsBos.get(i).getDataIndex() + "-" + defaultValueBos.get(j).getId())
                        );
                        array.add(jsonObject);
                    }
                }
            } catch (Exception e) {
                array.add(data);
                return;
            }
            return;
        }

        var propertiesJsonObject = JSONObject.parseObject(properties.toString(), Feature.OrderedField);
        array.add(data);
        var keySet = propertiesJsonObject.keySet();
        var collect = keySet.stream().map(key -> JSONObject.parseObject(JSONObject.toJSONString(propertiesJsonObject.get(key)), Feature.OrderedField)).collect(Collectors.toList());
        for (JSONObject item : collect) {
            problemJsonNew(array, item, Objects.nonNull(title) ? title.toString() : null);
        }
    }

    /**
     * crf 答案json 解析
     */
    public static List<CrfJsonValueBo> crfJsonAnalysis(String crfJson) {
        Map<String, String> map = new HashMap<>();
        if (Objects.isNull(crfJson) || "".equals(crfJson)) {
            return Collections.emptyList();
        }
        var jsonObject = JSONObject.parseObject(crfJson);
        if (jsonObject == null) {
            return Collections.emptyList();
        }
        var formStr = JSONObject.toJSONString(jsonObject.get("form"));
        if ("".equals(formStr)) {
            return Collections.emptyList();
        }
        var array = JSONObject.parseArray(formStr, CrfJsonValueBo.class);
        if (CollectionUtil.isNotEmpty(array)) {
            var bos = jsonArray(array);
            if (CollectionUtil.isEmpty(bos)) {
                return Collections.emptyList();
            }
            var collect = bos.stream().map(item -> {
                try {
                    var boList = JSONObject.parseArray(item.getText(), CrfJsonValueKeyBo.class);
                    if (CollectionUtil.isEmpty(boList)) {
                        return null;
                    }
                    List<CrfJsonValueBo> valueBos = new ArrayList<>();
                    var colKey = JSONObject.parseArray(boList.get(0).getColKey());
                    for (int i = 0; i < colKey.size(); i++) {
                        for (CrfJsonValueKeyBo crfJsonValueKeyBo : boList) {
                            var valueArray = JSONObject.parseArray(crfJsonValueKeyBo.getValue());
                            valueBos.add(
                                    new CrfJsonValueBo()
                                            .setKey(item.getKey() + "-" + colKey.get(i) + "-" + crfJsonValueKeyBo.getId())
                                            .setName(crfJsonValueKeyBo.getTitle())
                                            .setValue(crfJsonValueKeyBo.getValue())
                                            .setText(Objects.isNull(valueArray.get(i)) ? "-" : valueArray.get(i).toString())
                            );
                        }
                    }
                    return valueBos;
                } catch (Exception e) {
                    // log.info("【COPD-DATA-DOCTOR】 ★ crf json 解析异常");
                    return null;
                }
            }).filter(java.util.Objects::nonNull).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(collect)) {
                collect.forEach(bos::addAll);
            }
            return bos.stream().peek(item -> {
                item.setKeyPrefix(item.getKey());
            }).collect(Collectors.toList());
        }
        return Collections.emptyList();
    }


    /**
     * 递归解析
     *
     * @param array
     * @return
     */
    private static List<CrfJsonValueBo> jsonArray(List<CrfJsonValueBo> array) {
        var collect = array.stream().filter(item -> Objects.nonNull(item.getValue()) && !"".equals(item.getValue())).collect(Collectors.toList());
        if (CollectionUtil.isEmpty(collect)) {
            return collect;
        }
        List<CrfJsonValueBo> bos = new ArrayList<>();
        for (CrfJsonValueBo item : collect) {
            if (Objects.nonNull(item) && Objects.nonNull(item.getValue()) && !"".equals(item.getValue()) && !"null".equals(item.getValue())) {
                try {
                    var boList = JSONObject.parseArray(item.getValue(), CrfJsonValueBo.class).stream().filter(Objects::nonNull).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(boList)) {
                        bos.addAll(jsonArray(boList));
                    }
                } catch (Exception e) {
                    continue;
                }
            }
        }
        collect.addAll(bos);
        return collect;
    }

    public static void main(String[] args) {

        String a = "{\"form\":[{\"name\":\"基本信息\",\"key\":\"baseInfo_q2OjsA\",\"value\":[{\"key\":\"patientType\",\"name\":\"组别\",\"value\":null,\"text\":null},{\"key\":\"patientName\",\"name\":\"唯一识别码\",\"value\":\"78N6A\",\"text\":\"78N6A\"},{\"key\":\"patientGender\",\"name\":\"性别\",\"value\":[\"F\"],\"text\":\"女\"},{\"key\":\"patientNation\",\"name\":\"民族\",\"value\":null,\"text\":null},{\"key\":\"patientBirthmonth\",\"name\":\"出生年月\",\"value\":\"19930509\",\"text\":\"19930509\"},{\"key\":\"patientAge\",\"name\":\"年龄\",\"value\":\"29\",\"text\":\"29 岁\"},{\"key\":\"patientHeight\",\"name\":\"身高\",\"value\":\"167\",\"text\":\"167 cm\"},{\"key\":\"patientWeight\",\"name\":\"体重\",\"value\":\"54\",\"text\":\"54 kg\"},{\"key\":\"tips\",\"name\":\"备注\",\"value\":null,\"text\":null}]},{\"name\":\"个人史\",\"key\":\"smokeHistory_2NufHa\",\"value\":[{\"key\":\"smokeHis\",\"name\":\"吸烟史\",\"value\":[\"2\"],\"text\":\"戒烟超过1年\"},{\"key\":\"ApreviousSurgical\",\"name\":\"既往手术史\",\"value\":null,\"text\":null},{\"key\":\"previousSurgical\",\"name\":\"既往手术史-名称\",\"value\":null,\"text\":null},{\"key\":\"surgicalTime\",\"name\":\"手术时间(年月)\",\"value\":null,\"text\":null},{\"key\":\"AIndtoxins\",\"name\":\"工业毒物接触史\",\"value\":[\"0\"],\"text\":\"无\"},{\"key\":\"Indtoxins\",\"name\":\"工业毒物接触时长\",\"value\":null,\"text\":null},{\"key\":\"AdustRemoval\",\"name\":\"粉尘接触\",\"value\":[\"1\"],\"text\":\"有\"},{\"key\":\"dustRemoval\",\"name\":\"粉尘接触时长\",\"value\":null,\"text\":null},{\"key\":\"Aradioactive\",\"name\":\"放射性物质接触\",\"value\":[\"0\"],\"text\":\"无\"},{\"key\":\"radioactive\",\"name\":\"放射性物质接触时长\",\"value\":null,\"text\":null},{\"key\":\"foodAllergies\",\"name\":\"食物过敏史\",\"value\":[\"no\"],\"text\":\"无\"},{\"key\":\"drugAllergies\",\"name\":\"药物过敏史\",\"value\":[\"yes\"],\"text\":\"有\"},{\"key\":\"drugAllergiesother\",\"name\":\"其他过敏史(如花粉)\",\"value\":null,\"text\":null},{\"key\":\"familyAllergies\",\"name\":\"父母/祖父母是否有过敏\",\"value\":[\"yes\"],\"text\":\"有\"},{\"key\":\"ispollutionpollution\",\"name\":\"是否曾长期（大于五年）接触空气污染？\",\"value\":[\"no\"],\"text\":\"否\"},{\"key\":\"COVID\",\"name\":\"近1个月是否新冠感染\",\"value\":[\"1\"],\"text\":\"是\"}]},{\"name\":\"现病史(呼吸系统疾病)-近两周\",\"key\":\"CM_history_45UfkY\",\"value\":[{\"key\":\"nowDiseaseTime\",\"name\":\"当前疾病诊断时间\",\"value\":null,\"text\":null},{\"key\":\"isHot\",\"name\":\"发热\",\"value\":null,\"text\":null},{\"key\":\"nasalObstruction\",\"name\":\"鼻塞\",\"value\":null,\"text\":null},{\"key\":\"runNose\",\"name\":\"流涕\",\"value\":null,\"text\":null},{\"key\":\"frequency\",\"name\":\"咳嗽(没有感冒时)\",\"value\":null,\"text\":null},{\"key\":\"expectoration\",\"name\":\"咳痰量\",\"value\":null,\"text\":null},{\"key\":\"vomicaNt\",\"name\":\"咳脓痰\",\"value\":null,\"text\":null},{\"key\":\"pant\",\"name\":\"呼吸困难\",\"value\":null,\"text\":null},{\"key\":\"isBlood\",\"name\":\"咯血\",\"value\":null,\"text\":null},{\"key\":\"chestPain\",\"name\":\"胸痛\",\"value\":null,\"text\":null},{\"key\":\"otherqtzz\",\"name\":\"其他症状\",\"value\":null,\"text\":null},{\"key\":\"everyDayBreathBad\",\"name\":\"每天都有呼吸症状，在体力活动之后喘憋加重\",\"value\":null,\"text\":null},{\"key\":\"nowWorseAir\",\"name\":\"是否遇到刺激性气体等过敏性物质，或者情绪激动时，会引发呼吸道症状，但是之后能恢复正常\",\"value\":null,\"text\":null},{\"key\":\"NDDrugNameis\",\"name\":\"是否有药物治疗\",\"value\":null,\"text\":null},{\"key\":\"XBSHDetail\",\"name\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"value\":null,\"text\":null}]},{\"name\":\"现病史(其他呼吸系统疾病)\",\"key\":\"CMO_history_g3RXLT\",\"value\":[{\"key\":\"isHaveOtherBreath\",\"name\":\"是否有其他呼吸系统疾病\",\"value\":null,\"text\":null},{\"key\":\"otherDiseaseTime\",\"name\":\"诊断时间\",\"value\":null,\"text\":null},{\"key\":\"otherDiseaseName\",\"name\":\"疾病名称\",\"value\":null,\"text\":null},{\"key\":\"NDOBDrugNameis\",\"name\":\"是否有药物治疗\",\"value\":null,\"text\":null},{\"key\":\"XBSOBDetail\",\"name\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"value\":null,\"text\":null}]},{\"name\":\"现病史(非呼吸系统疾病)\",\"key\":\"CMF_history_q4Z5Br\",\"value\":[{\"key\":\"isHaveFBreath\",\"name\":\"是否有非呼吸系统疾病\",\"value\":null,\"text\":null},{\"key\":\"FDiseaseTime\",\"name\":\"诊断时间\",\"value\":null,\"text\":null},{\"key\":\"FDiseaseName\",\"name\":\"疾病名称\",\"value\":null,\"text\":null},{\"key\":\"NDFDrugNameis\",\"name\":\"是否有药物治疗\",\"value\":null,\"text\":null},{\"key\":\"XBSFDetail\",\"name\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"value\":null,\"text\":null}]},{\"name\":\"COPD过去1年内急性加重情况(仅COPD患者填写)\",\"key\":\"AEstatus_j4CJXr\",\"value\":[{\"key\":\"AETimes\",\"name\":\"AE总次数\",\"value\":null,\"text\":null},{\"key\":\"MidAE\",\"name\":\"中度AE\",\"value\":null,\"text\":null},{\"key\":\"highAE\",\"name\":\"重度AE\",\"value\":null,\"text\":null}]},{\"name\":\"HRCT\",\"key\":\"lungCTcheck_qHV_iL\",\"value\":[{\"key\":\"HRCTclear\",\"name\":\"是否已完成HRCT检查\",\"value\":null,\"text\":null},{\"key\":\"HRCTtime\",\"name\":\"HRCT检查日期\",\"value\":null,\"text\":null},{\"key\":\"HRCTcheck\",\"name\":\"HRCT诊断\",\"value\":null,\"text\":null},{\"key\":\"HRCTdescribe\",\"name\":\"HRCT描述\",\"value\":null,\"text\":null}]},{\"name\":\"胸部X线\",\"key\":\"lungCTcheckLower_DPV4NP\",\"value\":[{\"key\":\"chestXDateclear\",\"name\":\"是否已完成胸部X线检查\",\"value\":null,\"text\":null},{\"key\":\"chestXDate\",\"name\":\"胸部X线检查日期\",\"value\":null,\"text\":null},{\"key\":\"chestXresult\",\"name\":\"胸部X线检查结果\",\"value\":null,\"text\":null},{\"key\":\"chestXdescribe\",\"name\":\"胸部X线检查描述\",\"value\":null,\"text\":null}]},{\"name\":\"肺功能（哮喘与COPD必填）\",\"key\":\"lungFunction_inf_PtaUZb\",\"value\":[{\"key\":\"zqgkzjEight\",\"name\":\"是否停支气管舒张剂或激素类8h以上\",\"value\":null,\"text\":null},{\"key\":\"pulResult\",\"name\":\"舒张试验\",\"value\":null,\"text\":null},{\"key\":\"sdacJl\",\"name\":\"沙丁胺醇吸入剂量\",\"value\":null,\"text\":null},{\"key\":\"before\",\"name\":\"支气管扩张剂吸入前\",\"value\":[{\"predicted\":\"--\",\"real\":\"--\",\"percent\":\"--\",\"id\":\"0\",\"title\":\"FEV1%\"},{\"predicted\":\"--\",\"real\":\"--\",\"percent\":\"--\",\"id\":\"1\",\"title\":\"FEV1\"},{\"predicted\":\"--\",\"real\":\"--\",\"percent\":\"--\",\"id\":\"2\",\"title\":\"FVC\"},{\"predicted\":\"--\",\"real\":\"--\",\"percent\":\"--\",\"id\":\"3\",\"title\":\"FEV1/FVC\"}],\"text\":[{\"title\":\"FEV1%\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"predicted\",\"real\",\"percent\"],\"colName\":[\"预计值\",\"前次\",\"前/预\"],\"id\":\"0\"},{\"title\":\"FEV1\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"predicted\",\"real\",\"percent\"],\"colName\":[\"预计值\",\"前次\",\"前/预\"],\"id\":\"1\"},{\"title\":\"FVC\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"predicted\",\"real\",\"percent\"],\"colName\":[\"预计值\",\"前次\",\"前/预\"],\"id\":\"2\"},{\"title\":\"FEV1/FVC\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"predicted\",\"real\",\"percent\"],\"colName\":[\"预计值\",\"前次\",\"前/预\"],\"id\":\"3\"}]},{\"key\":\"after\",\"name\":\"支气管扩张剂吸入后\",\"value\":[{\"real\":\"--\",\"percent\":\"--\",\"better\":\"--\",\"id\":\"0\",\"title\":\"FEV1%\"},{\"real\":\"--\",\"percent\":\"--\",\"better\":\"--\",\"id\":\"1\",\"title\":\"FEV1\"},{\"real\":\"--\",\"percent\":\"--\",\"better\":\"--\",\"id\":\"2\",\"title\":\"FVC\"},{\"real\":\"--\",\"percent\":\"--\",\"better\":\"--\",\"id\":\"3\",\"title\":\"FEV1/FVC\"}],\"text\":[{\"title\":\"FEV1%\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"real\",\"percent\",\"better\"],\"colName\":[\"后次\",\"后/预\",\"改善率\"],\"id\":\"0\"},{\"title\":\"FEV1\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"real\",\"percent\",\"better\"],\"colName\":[\"后次\",\"后/预\",\"改善率\"],\"id\":\"1\"},{\"title\":\"FVC\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"real\",\"percent\",\"better\"],\"colName\":[\"后次\",\"后/预\",\"改善率\"],\"id\":\"2\"},{\"title\":\"FEV1/FVC\",\"value\":[\"--\",\"--\",\"--\"],\"colKey\":[\"real\",\"percent\",\"better\"],\"colName\":[\"后次\",\"后/预\",\"改善率\"],\"id\":\"3\"}]}]}],\"crfId\":\"1828353427575558144\",\"answer\":{\"patientName\":\"78N6A\",\"patientGender\":[\"F\"],\"patientPhone\":\"\",\"patientBirth\":\"\",\"patientHeight\":\"167\",\"patientWeight\":\"54\",\"patientAge\":\"29\",\"patientBirthmonth\":\"19930509\",\"smokeHis\":[\"2\"],\"AIndtoxins\":[\"0\"],\"AdustRemoval\":[\"1\"],\"Aradioactive\":[\"0\"],\"foodAllergies\":[\"no\"],\"drugAllergies\":[\"yes\"],\"familyAllergies\":[\"yes\"],\"ispollutionpollution\":[\"no\"],\"COVID\":[\"1\"],\"before\":[{\"id\":\"0\",\"title\":\"FEV1%\"},{\"id\":\"1\",\"title\":\"FEV1\"},{\"id\":\"2\",\"title\":\"FVC\"},{\"id\":\"3\",\"title\":\"FEV1/FVC\"}],\"after\":[{\"id\":\"0\",\"title\":\"FEV1%\"},{\"id\":\"1\",\"title\":\"FEV1\"},{\"id\":\"2\",\"title\":\"FVC\"},{\"id\":\"3\",\"title\":\"FEV1/FVC\"}]}}";
        String form = "{\"displayType\":\"column\",\"code\":\"patient_inf\",\"name\":\"患者录入表单\",\"type\":\"object\",\"properties\":{\"baseInfo_q2OjsA\":{\"type\":\"object\",\"description\":\"\",\"title\":\"基本信息\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"baseInfo\"},\"properties\":{\"radio_pZJcYQ\":{\"title\":\"组别\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"patientType\"},\"options\":[{\"id\":\"0\",\"name\":\"COPD\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"支气管哮喘\",\"code\":\"2\"},{\"id\":\"2\",\"name\":\"支气管扩张\",\"code\":\"3\"},{\"id\":\"3\",\"name\":\"上呼吸道感染\",\"code\":\"4\"},{\"id\":\"4\",\"name\":\"下呼吸道感染\",\"code\":\"5\"},{\"id\":\"5\",\"name\":\"慢性支气管炎\",\"code\":\"7\"},{\"id\":\"6\",\"name\":\"咳嗽\",\"code\":\"8\"},{\"id\":\"7\",\"name\":\"健康人\",\"code\":\"6\"}],\"enum\":[\"1\",\"2\",\"3\",\"4\",\"5\",\"7\",\"8\",\"6\"],\"enumNames\":[\"COPD\",\"支气管哮喘\",\"支气管扩张\",\"上呼吸道感染\",\"下呼吸道感染\",\"慢性支气管炎\",\"咳嗽\",\"健康人\"]},\"input_eI5d5N\":{\"title\":\"唯一识别码\",\"type\":\"string\",\"questionType\":\"TEXT\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"patientName\"},\"rangeLength\":{\"min\":1,\"max\":5},\"min\":1,\"max\":5,\"show\":{\"questions\":[],\"options\":[]}},\"radio_2BZqLQ\":{\"title\":\"性别\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"patientGender\"},\"options\":[{\"id\":\"0\",\"name\":\"男\",\"code\":\"M\"},{\"id\":\"1\",\"name\":\"女\",\"code\":\"F\"}],\"enum\":[\"M\",\"F\"],\"enumNames\":[\"男\",\"女\"]},\"input_1vS3p_\":{\"title\":\"民族\",\"type\":\"string\",\"questionType\":\"TEXT\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"patientNation\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"input_BBOEYs\":{\"title\":\"出生年月\",\"type\":\"string\",\"questionType\":\"TEXT\",\"description\":\"如1990年12月，请输入199012\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"patientBirthmonth\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"number_GfRINe\":{\"title\":\"年龄\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"patientAge\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"岁\",\"min\":0,\"max\":0,\"required\":true},\"number_xhJFdf\":{\"title\":\"身高\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"patientHeight\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"cm\",\"min\":0,\"max\":0,\"required\":true},\"number_2NItPR\":{\"title\":\"体重\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"patientWeight\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"kg\",\"min\":0,\"max\":0,\"required\":true},\"textarea_InDWcs\":{\"title\":\"备注\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"tips\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"smokeHistory_2NufHa\":{\"type\":\"object\",\"description\":\"\",\"title\":\"个人史\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"smokeHistory\"},\"required\":true,\"properties\":{\"radio_wxrmgo\":{\"title\":\"吸烟史\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"smokeHis\"},\"options\":[{\"id\":\"0\",\"name\":\"正在吸烟/戒烟未满1年\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"戒烟超过1年\",\"code\":\"2\"},{\"id\":\"2\",\"name\":\"从不吸烟\",\"code\":\"3\"}],\"enum\":[\"1\",\"2\",\"3\"],\"enumNames\":[\"正在吸烟/戒烟未满1年\",\"戒烟超过1年\",\"从不吸烟\"]},\"radio_PjH1Z_\":{\"title\":\"既往手术史\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"ApreviousSurgical\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"input_ELJeHV\":{\"title\":\"既往手术史-名称\",\"type\":\"string\",\"questionType\":\"TEXT\",\"description\":\"\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"previousSurgical\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"input_meEqwd\":{\"title\":\"手术时间(年月)\",\"type\":\"string\",\"questionType\":\"TEXT\",\"description\":\"如1990年12月，请输入199012\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"surgicalTime\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"radio_Sk6gDE\":{\"title\":\"工业毒物接触史\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"AIndtoxins\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"number_ML7RDe\":{\"title\":\"工业毒物接触时长\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"description\":\"\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"Indtoxins\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"年\",\"min\":0,\"max\":0},\"radio_ksSICF\":{\"title\":\"粉尘接触\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"AdustRemoval\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"number_0Pj-iN\":{\"title\":\"粉尘接触时长\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"description\":\"\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"dustRemoval\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"年\",\"min\":0,\"max\":0},\"radio_IFLN4Y\":{\"title\":\"放射性物质接触\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"Aradioactive\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"number_xhGgoA\":{\"title\":\"放射性物质接触时长\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"description\":\"\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"radioactive\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"年\",\"min\":0,\"max\":0},\"radio_CoFBNM\":{\"title\":\"食物过敏史\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"foodAllergies\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_9lmZfk\":{\"title\":\"药物过敏史\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"drugAllergies\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"input_-LeeUZ\":{\"title\":\"其他过敏史(如花粉)\",\"type\":\"string\",\"questionType\":\"TEXT\",\"required\":false,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"drugAllergiesother\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"radio_VHyLEJ\":{\"title\":\"父母/祖父母是否有过敏\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"过敏性鼻炎、湿疹皮肤过敏、食物过敏等\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"familyAllergies\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_nnqvcY\":{\"title\":\"是否曾长期（大于五年）接触空气污染？\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"空气污染包含粉尘、刺激气体、麻将室茶馆室二手烟，或常年工作在化工、纺织车间等有空气污染的环境\",\"required\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"ispollutionpollution\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"radio_5veO9S\":{\"title\":\"近1个月是否新冠感染\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"鼻或咽拭子抗原/聚合酶链反应 (PCR)检测阳性\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"COVID\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"2\"}],\"enum\":[\"1\",\"2\"],\"enumNames\":[\"是\",\"否\"],\"required\":true}}},\"CM_history_45UfkY\":{\"type\":\"object\",\"title\":\"现病史(呼吸系统疾病)-近两周\",\"description\":\"\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"CM_history\"},\"properties\":{\"date_SZbSoa\":{\"title\":\"当前疾病诊断时间\",\"type\":\"string\",\"format\":\"date\",\"questionType\":\"DATE\",\"widget\":\"CrfDatePick\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"nowDiseaseTime\"}},\"radio_RRlBks\":{\"title\":\"发热\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"isHot\"},\"options\":[{\"id\":\"0\",\"name\":\"37.3~38℃\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"38.1~39℃\",\"code\":\"2\"},{\"id\":\"2\",\"name\":\"39.1~41℃\",\"code\":\"3\"},{\"id\":\"3\",\"name\":\"≥41℃\",\"code\":\"4\"}],\"enum\":[\"1\",\"2\",\"3\",\"4\"],\"enumNames\":[\"37.3~38℃\",\"38.1~39℃\",\"39.1~41℃\",\"≥41℃\"]},\"radio_9Wf2Pa\":{\"title\":\"鼻塞\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"nasalObstruction\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_BKQTvO\":{\"title\":\"流涕\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"runNose\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_3oCGKT\":{\"title\":\"咳嗽(没有感冒时)\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"VS\",\"dataId\":\"frequency\"},\"options\":[{\"id\":\"0\",\"name\":\"无明显咳嗽\",\"code\":\"neverCough\"},{\"id\":\"1\",\"name\":\"早晨居多\",\"code\":\"mornMore\"},{\"id\":\"2\",\"name\":\"夜间居多\",\"code\":\"nightMore\"},{\"id\":\"3\",\"name\":\"全天偶有咳嗽\",\"code\":\"occasionalCough\"},{\"id\":\"4\",\"name\":\"咳嗽频繁\",\"code\":\"frequentCough\"}],\"enum\":[\"neverCough\",\"mornMore\",\"nightMore\",\"occasionalCough\",\"frequentCough\"],\"enumNames\":[\"无明显咳嗽\",\"早晨居多\",\"夜间居多\",\"全天偶有咳嗽\",\"咳嗽频繁\"]},\"radio_4MD3cs\":{\"title\":\"咳痰量\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"VS\",\"dataId\":\"expectoration\"},\"options\":[{\"id\":\"0\",\"name\":\"无明显咳痰\",\"code\":\"neverExpectorate\"},{\"id\":\"1\",\"name\":\"咳痰10ml以下\",\"code\":\"belowTen\"},{\"id\":\"2\",\"name\":\"咳痰10ml以上\",\"code\":\"aboveTen\"}],\"enum\":[\"neverExpectorate\",\"belowTen\",\"aboveTen\"],\"enumNames\":[\"无明显咳痰\",\"咳痰10ml以下\",\"咳痰10ml以上\"]},\"radio_svNkPO\":{\"title\":\"咳脓痰\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"vomicaNt\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_EKR86w\":{\"title\":\"呼吸困难\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"VS\",\"dataId\":\"pant\"},\"options\":[{\"id\":\"0\",\"name\":\"无\",\"code\":\"neverPant\"},{\"id\":\"1\",\"name\":\"剧烈活动时出现\",\"code\":\"occasionallyPant\"},{\"id\":\"2\",\"name\":\"日常活动时发生，休息时不发生\",\"code\":\"whenActive\"},{\"id\":\"3\",\"name\":\"喘息不能平躺，影响休息与活动\",\"code\":\"alwaysPant\"}],\"enum\":[\"neverPant\",\"occasionallyPant\",\"whenActive\",\"alwaysPant\"],\"enumNames\":[\"无\",\"剧烈活动时出现\",\"日常活动时发生，休息时不发生\",\"喘息不能平躺，影响休息与活动\"]},\"radio_H-oWj_\":{\"title\":\"咯血\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"isBlood\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"radio_iFybYL\":{\"title\":\"胸痛\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"chestPain\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"input_Tv2xuD\":{\"title\":\"其他症状\",\"type\":\"string\",\"questionType\":\"TEXT\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"otherqtzz\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"radio_pMRb1u\":{\"title\":\"每天都有呼吸症状，在体力活动之后喘憋加重\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"喘息，咳嗽，呼吸困难，胸闷或疼痛\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"everyDayBreathBad\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"radio_6FzCNg\":{\"title\":\"是否遇到刺激性气体等过敏性物质，或者情绪激动时，会引发呼吸道症状，但是之后能恢复正常\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"description\":\"喘息，咳嗽，呼吸困难，胸闷或疼痛\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"nowWorseAir\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"radio_mN0BSK\":{\"title\":\"是否有药物治疗\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"NDDrugNameis\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"textarea_VcEcqT\":{\"title\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"description\":\"填写示例：药物1，2mg，口服，每天2次\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"XBSHDetail\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"CMO_history_g3RXLT\":{\"type\":\"object\",\"description\":\"\",\"title\":\"现病史(其他呼吸系统疾病)\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"CMO_history\"},\"properties\":{\"radio_gmS0wy\":{\"title\":\"是否有其他呼吸系统疾病\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"isHaveOtherBreath\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"date_SZbSoa\":{\"title\":\"诊断时间\",\"type\":\"string\",\"format\":\"date\",\"questionType\":\"DATE\",\"widget\":\"CrfDatePick\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"otherDiseaseTime\"}},\"input_sXY8NP\":{\"title\":\"疾病名称\",\"type\":\"string\",\"questionType\":\"TEXT\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"otherDiseaseName\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"radio_bsMHh_\":{\"title\":\"是否有药物治疗\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"NDOBDrugNameis\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"textarea_ju4cIb\":{\"title\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"description\":\"填写示例：药物1，2mg，口服，每天2次\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"XBSOBDetail\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"CMF_history_q4Z5Br\":{\"type\":\"object\",\"description\":\"\",\"title\":\"现病史(非呼吸系统疾病)\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"CMF_history\"},\"properties\":{\"radio_gmS0wy\":{\"title\":\"是否有非呼吸系统疾病\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"isHaveFBreath\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"有\",\"无\"]},\"date_SZbSoa\":{\"title\":\"诊断时间\",\"type\":\"string\",\"format\":\"date\",\"questionType\":\"DATE\",\"widget\":\"CrfDatePick\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"FDiseaseTime\"}},\"input_sXY8NP\":{\"title\":\"疾病名称\",\"type\":\"string\",\"questionType\":\"TEXT\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"FDiseaseName\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"radio_Arisrv\":{\"title\":\"是否有药物治疗\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"NDFDrugNameis\"},\"options\":[{\"id\":\"0\",\"name\":\"有\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"无\",\"code\":\"0\"}],\"enum\":[\"1\",\"0\"],\"enumNames\":[\"有\",\"无\"]},\"textarea_FrL9N0\":{\"title\":\"治疗药物详情（药物名称/剂量/给药途径/频率）\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"description\":\"填写示例：药物1，2mg，口服，每天2次\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"XBSFDetail\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"AEstatus_j4CJXr\":{\"type\":\"object\",\"description\":\"\",\"title\":\"COPD过去1年内急性加重情况(仅COPD患者填写)\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"AEstatus\"},\"properties\":{\"number_d3YdWP\":{\"title\":\"AE总次数\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"AETimes\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"次\",\"min\":0,\"max\":0},\"number_s3YyoQ\":{\"title\":\"中度AE\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"description\":\"使用短效支气管扩张剂和抗生素治疗和/或口服糖皮质激素\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"MidAE\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"次\",\"min\":0,\"max\":0},\"number_b5HNLY\":{\"title\":\"重度AE\",\"type\":\"number\",\"questionType\":\"NUMBER\",\"widget\":\"CrfInputNumber\",\"description\":\"需要住院或急诊、ICU 治疗\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"highAE\"},\"range\":{\"min\":0,\"max\":0},\"unit\":\"次\",\"min\":0,\"max\":0}}},\"lungCTcheck_qHV_iL\":{\"type\":\"object\",\"description\":\"\",\"title\":\"HRCT\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"lungCTcheck\"},\"properties\":{\"radio_eQ7i2_\":{\"title\":\"是否已完成HRCT检查\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"HRCTclear\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"date_eLLo3o\":{\"title\":\"HRCT检查日期\",\"type\":\"string\",\"format\":\"date\",\"questionType\":\"DATE\",\"widget\":\"CrfDatePick\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"HRCTtime\"}},\"input_REgSRb\":{\"title\":\"HRCT诊断\",\"type\":\"string\",\"questionType\":\"TEXT\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"HRCTcheck\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"textarea_1nJhSZ\":{\"title\":\"HRCT描述\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"HRCTdescribe\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"lungCTcheckLower_DPV4NP\":{\"type\":\"object\",\"description\":\"\",\"title\":\"胸部X线\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"lungCTcheckLower\"},\"properties\":{\"radio_F3h09O\":{\"title\":\"是否已完成胸部X线检查\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"chestXDateclear\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"date_OPurFU\":{\"title\":\"胸部X线检查日期\",\"type\":\"string\",\"format\":\"date\",\"questionType\":\"DATE\",\"widget\":\"CrfDatePick\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"chestXDate\"}},\"input_GhAtyb\":{\"title\":\"胸部X线检查结果\",\"type\":\"string\",\"questionType\":\"TEXT\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"chestXresult\"},\"rangeLength\":{\"min\":0,\"max\":200},\"min\":0,\"max\":200},\"textarea_V0f1S3\":{\"title\":\"胸部X线检查描述\",\"type\":\"string\",\"format\":\"textarea\",\"questionType\":\"TEXTAREA\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"chestXdescribe\"},\"rangeLength\":{\"min\":0,\"max\":400},\"min\":0,\"max\":400}}},\"lungFunction_inf_PtaUZb\":{\"type\":\"object\",\"description\":\"\",\"title\":\"肺功能（哮喘与COPD必填）\",\"questionType\":\"TEMPLATE\",\"lucaBanpickDataId\":true,\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"lungFunction_inf\"},\"properties\":{\"radio_eiDPtt\":{\"title\":\"是否停支气管舒张剂或激素类8h以上\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"zqgkzjEight\"},\"options\":[{\"id\":\"0\",\"name\":\"是\",\"code\":\"yes\"},{\"id\":\"1\",\"name\":\"否\",\"code\":\"no\"}],\"enum\":[\"yes\",\"no\"],\"enumNames\":[\"是\",\"否\"]},\"radio_TwQ-dW\":{\"title\":\"舒张试验\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"pulResult\"},\"options\":[{\"id\":\"0\",\"name\":\"阳性\",\"code\":\"positive\"},{\"id\":\"1\",\"name\":\"阴性\",\"code\":\"negative\"}],\"enum\":[\"positive\",\"negative\"],\"enumNames\":[\"阳性\",\"阴性\"]},\"radio_g2D3JC\":{\"title\":\"沙丁胺醇吸入剂量\",\"type\":\"string\",\"widget\":\"radio\",\"questionType\":\"RADIO\",\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"MH\",\"dataId\":\"sdacJl\"},\"options\":[{\"id\":\"0\",\"name\":\"100μg\",\"code\":\"1\"},{\"id\":\"1\",\"name\":\"200μg\",\"code\":\"2\"},{\"id\":\"2\",\"name\":\"400μg\",\"code\":\"3\"},{\"id\":\"3\",\"name\":\"不详\",\"code\":\"4\"}],\"enum\":[\"1\",\"2\",\"3\",\"4\"],\"enumNames\":[\"100μg\",\"200μg\",\"400μg\",\"不详\"]},\"newTabel_CeMggW\":{\"title\":\"支气管扩张剂吸入前\",\"type\":\"array\",\"questionType\":\"CRFTABLE\",\"widget\":\"CrfTable\",\"items\":{\"type\":\"object\",\"properties\":{}},\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"VS\",\"dataId\":\"before\"},\"columns\":[{\"id\":\"0\",\"title\":\"指标\",\"dataIndex\":\"title\",\"type\":\"TEXT\"},{\"id\":\"1\",\"title\":\"预计值\",\"dataIndex\":\"predicted\",\"type\":\"INPUT\"},{\"id\":\"2\",\"title\":\"前次\",\"dataIndex\":\"real\",\"type\":\"INPUT\"},{\"id\":\"3\",\"title\":\"前/预\",\"dataIndex\":\"percent\",\"type\":\"INPUT\"}],\"defaultValue\":[{\"id\":\"0\",\"title\":\"FEV1%\"},{\"id\":\"1\",\"title\":\"FEV1\"},{\"id\":\"2\",\"title\":\"FVC\"},{\"id\":\"3\",\"title\":\"FEV1/FVC\"}],\"description\":\"\"},\"newTabel_U738QJ\":{\"title\":\"支气管扩张剂吸入后\",\"type\":\"array\",\"questionType\":\"CRFTABLE\",\"widget\":\"CrfTable\",\"items\":{\"type\":\"object\",\"properties\":{}},\"BindData\":{\"dataType\":\"CUSTOMIZE\",\"choiceField\":\"DM\",\"dataId\":\"after\"},\"columns\":[{\"id\":\"0\",\"title\":\"指标\",\"dataIndex\":\"title\",\"type\":\"TEXT\"},{\"id\":\"1\",\"title\":\"后次\",\"dataIndex\":\"real\",\"type\":\"INPUT\"},{\"id\":\"2\",\"title\":\"后/预\",\"dataIndex\":\"percent\",\"type\":\"INPUT\"},{\"id\":\"3\",\"title\":\"改善率\",\"dataIndex\":\"better\",\"type\":\"INPUT\"}],\"defaultValue\":[{\"id\":\"0\",\"title\":\"FEV1%\"},{\"id\":\"1\",\"title\":\"FEV1\"},{\"id\":\"2\",\"title\":\"FVC\"},{\"id\":\"3\",\"title\":\"FEV1/FVC\"}],\"description\":\"\"}}}}}";

        System.out.println(JSONUtil.toJsonStr(crfProblemJsonAnalysisNew(form)));
        System.out.println(JSONUtil.toJsonStr(crfJsonAnalysis(a)));

        System.out.println(parseCrfProblemAndAnswer(form, a));
    }

}
